<?php
// Filename: includes/functions.php
// Description: توابع پایه و کمکی (ثبت‌نام، محاسبات، تلگرام و دیپلماسی)

require_once __DIR__ . '/db_connect.php';

// 1. شروع ثبت‌نام (ایجاد رکورد اولیه)
function initPlayer($user_id) {
    global $conn;
    $stmt = $conn->prepare("INSERT INTO players (user_id, username, capital_x, capital_y) VALUES (?, 'Unnamed', 0, 0)");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
}

// 2. تکمیل ثبت‌نام (بعد از دریافت نام)

// تابع کمکی برای پیدا کردن یک مختصات خالی و امن
function findSafeSpawnLocation() {
    global $conn;
    
    // 1. محاسبه شعاع فعلی بازی بر اساس تعداد بازیکنان
    // فرمول: هر بازیکن به حدود 50 خانه فضا نیاز دارد
    $res = $conn->query("SELECT COUNT(*) as cnt FROM players WHERE capital_x > 0");
    $count = $res->fetch_assoc()['cnt'];
    
    // حداقل شعاع 20، و با افزایش بازیکنان شعاع زیاد می‌شود
    $radius = max(20, ceil(sqrt($count * 50))); 
    $max_limit = min($radius, MAP_SIZE); 

    // 2. تلاش برای پیدا کردن جای خالی (تا 50 بار تلاش)
    for ($i = 0; $i < 50; $i++) {
        $x = rand(1, $max_limit);
        $y = rand(1, $max_limit);
        
        // چک کنیم آیا اینجا قبلاً توسط کسی اشغال شده (قلعه یا شهر)؟
        $check = $conn->query("SELECT owner_id FROM map_tiles WHERE x=$x AND y=$y AND owner_id IS NOT NULL");
        if ($check->num_rows > 0) continue; // اشغال است، دوباره تلاش کن

        // چک کنیم آیا خیلی نزدیک به یک قلعه دیگر است؟ (فاصله امن: 3 خانه)
        $distCheck = $conn->query("SELECT x FROM map_tiles WHERE type='capital' AND x BETWEEN ".($x-3)." AND ".($x+3)." AND y BETWEEN ".($y-3)." AND ".($y+3));
        if ($distCheck->num_rows > 0) continue; // خیلی نزدیک همسایه است
        
        return ['x' => $x, 'y' => $y];
    }
    
    // اگر جای امن پیدا نشد، یه جای رندوم بده (حالت اضطراری)
    return ['x' => rand(1, MAP_SIZE), 'y' => rand(1, MAP_SIZE)];
}

// 2. تکمیل ثبت‌نام (نسخه اصلاح شده برای جمعیت بالا)
function completeRegistration($user_id, $name) {
    global $conn;
    
    // *** استفاده از تابع اسپان هوشمند ***
    $loc = findSafeSpawnLocation();
    $x = $loc['x'];
    $y = $loc['y'];
    
    $name = htmlspecialchars(substr($name, 0, 30));
    
    // آپدیت بازیکن
    $stmt = $conn->prepare("UPDATE players SET username=?, capital_x=?, capital_y=? WHERE user_id=?");
    $stmt->bind_param("siii", $name, $x, $y, $user_id);
    $stmt->execute();
    
    // منابع اولیه
    $res = STARTING_RESOURCES;
    $stmt = $conn->prepare("INSERT INTO player_resources (user_id, food, wood, stone, gold) VALUES (?, ?, ?, ?, ?)");
    $stmt->bind_param("iiiii", $user_id, $res['food'], $res['wood'], $res['stone'], $res['gold']);
    $stmt->execute();
    
    // ثبت پایتخت
    $garrison = json_encode(['swordsman' => 10]);
    $stmt = $conn->prepare("INSERT INTO map_tiles (x, y, owner_id, type, garrison) VALUES (?, ?, ?, 'capital', ?) ON DUPLICATE KEY UPDATE owner_id=?, type='capital', garrison=?");
    $stmt->bind_param("iiisis", $x, $y, $user_id, $garrison, $user_id, $garrison);
    $stmt->execute();
    
    // ایجاد چند خانه منابع اطراف پایتخت (برای اینکه بازیکن در بیابان نباشد)
    // مثلاً 1 جنگل و 1 کوه در نزدیکی ایجاد می‌کنیم
    $conn->query("INSERT IGNORE INTO map_tiles (x, y, type) VALUES (".($x+1).", $y, 'forest')");
    $conn->query("INSERT IGNORE INTO map_tiles (x, y, type) VALUES ($x, ".($y+1).", 'mountain')");

    // ژنرال اولیه
    $troops = json_encode(['swordsman' => 20]);
    $stmt = $conn->prepare("INSERT INTO generals (user_id, name, x, y, mp, max_mp, troops) VALUES (?, 'ژنرال ارشد', ?, ?, ?, ?, ?)");
    $mp = BASE_MOVEMENT_POINTS;
    $stmt->bind_param("iiiiis", $user_id, $x, $y, $mp, $mp, $troops);
    $stmt->execute();
    
    return ['x' => $x, 'y' => $y, 'name' => $name];
}

// دریافت نام بازیکن
function getPlayerName($user_id) {
    global $conn;
    $res = $conn->query("SELECT username FROM players WHERE user_id = $user_id");
    if ($row = $res->fetch_assoc()) {
        return $row['username'] ?: "بازیکن $user_id";
    }
    return "ناشناس";
}

// محاسبه فاصله
function calculateDistance($x1, $y1, $x2, $y2) {
    return abs($x1 - $x2) + abs($y1 - $y2);
}

// هزینه حرکت
function getTerrainCost($type) {
    switch ($type) {
        case 'forest': return 2;
        case 'mountain': return 3;
        case 'water': return 999;
        case 'capital': return 1;
        default: return 1;
    }
}

// ارسال پیام تلگرام
function sendMessage($chat_id, $text, $keyboard = null) {
    $data = ['chat_id' => $chat_id, 'text' => $text, 'parse_mode' => 'HTML'];
    if ($keyboard) $data['reply_markup'] = json_encode($keyboard);
    $ch = curl_init("https://api.telegram.org/bot" . TELEGRAM_BOT_TOKEN . "/sendMessage");
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $result = curl_exec($ch);
    curl_close($ch);
    return $result;
}

// پاسخ به کال‌بک (حذف لودینگ)
function answerCallbackQuery($callback_query_id, $text = null) {
    $data = ['callback_query_id' => $callback_query_id];
    if ($text) {
        $data['text'] = $text;
        $data['show_alert'] = true;
    }
    $ch = curl_init("https://api.telegram.org/bot" . TELEGRAM_BOT_TOKEN . "/answerCallbackQuery");
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_exec($ch);
    curl_close($ch);
}

// بررسی اتحاد
function checkAlliance($user1, $user2) {
    global $conn;
    if ($user1 == $user2) return false;

    $sql = "SELECT id FROM diplomacy 
            WHERE status='ally' AND 
            ((requester_id=$user1 AND receiver_id=$user2) OR (requester_id=$user2 AND receiver_id=$user1)) 
            LIMIT 1";
    $result = $conn->query($sql);
    return ($result->num_rows > 0);
}

// *** (جدید) محاسبه نرخ تولید روزانه ***
function calculatePlayerProduction($user_id) {
    global $conn;
    
    // دریافت سطح ساختمان‌ها
    $bRes = $conn->query("SELECT type, level FROM buildings WHERE user_id=$user_id");
    $levels = [];
    while($b = $bRes->fetch_assoc()) $levels[$b['type']] = $b['level'];
    
    $farmLvl = $levels['farm'] ?? 0;
    $mineLvl = $levels['mine'] ?? 0;
    $lumberLvl = $levels['lumber'] ?? 0;
    
    if (!defined('BUILDING_DATA')) return ['food'=>0, 'wood'=>0, 'stone'=>0];

    $prodFood = ($farmLvl > 0 ? $farmLvl * BUILDING_DATA['farm']['prod_rate'] : 10) * 24;
    $prodStone = ($mineLvl > 0 ? $mineLvl * BUILDING_DATA['mine']['prod_rate'] : 5) * 24;
    $prodWood = ($lumberLvl > 0 ? $lumberLvl * BUILDING_DATA['lumber']['prod_rate'] : 10) * 24;
    
    return ['food' => $prodFood, 'wood' => $prodWood, 'stone' => $prodStone];
}

?>