<?php
// Filename: includes/game_logic.php
// Description: تمام منطق‌های تعاملی بازی (نقشه، ارتش، شهر، جاسوسی، انتقال نیرو، رتبه‌بندی، دیپلماسی)

require_once __DIR__ . '/db_connect.php';
require_once __DIR__ . '/functions.php';

// ---------------------------------------------------------
// 1. توابع مربوط به نقشه (Map Functions)
// ---------------------------------------------------------

function getPlayerColor($user_id) {
    $colors = ['🟥', '🟧', '🟨', '🟩', '🟦', '🟪', '🟫', '⬛', '🔶'];
    return $colors[$user_id % count($colors)];
}

function getRelationColor($viewer_id, $owner_id) {
    if ($viewer_id == $owner_id) return '🟦'; 
    if (checkAlliance($viewer_id, $owner_id)) return '🟩'; 
    return '🟥'; 
}

function handleShowMap($user_id, $chat_id, $center_x, $center_y) {
    global $conn;
    
    $range_x = 2; 
    $range_y = 5; 
    $territory_radius = 1; 
    
    $min_x = max(1, $center_x - $range_x);
    $max_x = min(MAP_SIZE, $center_x + $range_x);
    $min_y = max(1, $center_y - $range_y);
    $max_y = min(MAP_SIZE, $center_y + $range_y);
    
    // دریافت تایل‌ها
    $tiles = [];
    $sql = "SELECT * FROM map_tiles WHERE x BETWEEN $min_x AND $max_x AND y BETWEEN $min_y AND $max_y";
    $result = $conn->query($sql);
    while($row = $result->fetch_assoc()) $tiles[$row['y']][$row['x']] = $row;

    // دریافت ژنرال‌ها (پشتیبانی از چند آیکون)
    $generals = [];
    $genSql = "SELECT * FROM generals WHERE x BETWEEN $min_x AND $max_x AND y BETWEEN $min_y AND $max_y";
    $genResult = $conn->query($genSql);
    while($gen = $genResult->fetch_assoc()) {
        $generals[$gen['y']][$gen['x']][] = $gen;
    }
    
    $pendingMoves = [];
    $ordSql = "SELECT target_x, target_y FROM turn_orders WHERE user_id = $user_id 
               AND target_x BETWEEN $min_x AND $max_x AND target_y BETWEEN $min_y AND $max_y";
    $ordResult = $conn->query($ordSql);
    while($ord = $ordResult->fetch_assoc()) $pendingMoves[$ord['target_y']][$ord['target_x']] = true;

    $cap_min_x = max(1, $min_x - $territory_radius); $cap_max_x = min(MAP_SIZE, $max_x + $territory_radius);
    $cap_min_y = max(1, $min_y - $territory_radius); $cap_max_y = min(MAP_SIZE, $max_y + $territory_radius);
    $capResult = $conn->query("SELECT x, y, owner_id FROM map_tiles WHERE type='capital' AND owner_id IS NOT NULL AND x BETWEEN $cap_min_x AND $cap_max_x AND y BETWEEN $cap_min_y AND $cap_max_y");
    
    $territoryMap = []; 
    while($cap = $capResult->fetch_assoc()) {
        for ($tx = $cap['x'] - $territory_radius; $tx <= $cap['x'] + $territory_radius; $tx++) {
            for ($ty = $cap['y'] - $territory_radius; $ty <= $cap['y'] + $territory_radius; $ty++) {
                if ($tx >= 1 && $tx <= MAP_SIZE && $ty >= 1 && $ty <= MAP_SIZE) {
                    $territoryMap[$ty][$tx] = $cap['owner_id'];
                }
            }
        }
    }
    
    $myColor = getPlayerColor($user_id);
    $keyboard = [];
    $map_display = "🗺 <b>نقشه منطقه ($center_x, $center_y)</b>\n";
    $map_display .= "راهنما: 🟦 شما | 🟩 متحد | 🟥 دشمن | 🛡️ ارتش متحد | 🎯 هدف\n\n";
    
    for ($y = $min_y; $y <= $max_y; $y++) {
        $row_btns = [];
        for ($x = $min_x; $x <= $max_x; $x++) {
            $tile = $tiles[$y][$x] ?? ['type' => 'plain', 'owner_id' => null];
            $tileGenerals = $generals[$y][$x] ?? [];
            $territoryOwner = $territoryMap[$y][$x] ?? null;
            $isPending = isset($pendingMoves[$y][$x]); 
            
            $icon = '⬜'; 
            
            // نوع زمین
            if ($tile['type'] == 'forest') $icon = '🌲';
            elseif ($tile['type'] == 'mountain') $icon = '⛰';
            elseif ($tile['type'] == 'water') $icon = '🌊';
            elseif ($tile['type'] == 'capital') $icon = '🏯';

            // رنگ قلمرو
            if ($territoryOwner && $tile['type'] != 'capital') {
                $icon = getRelationColor($user_id, $territoryOwner);
            }
            
            // قلعه
            if ($tile['type'] == 'capital' && $tile['owner_id']) {
                $color = getRelationColor($user_id, $tile['owner_id']);
                $icon = ($tile['owner_id'] == $user_id) ? '🏰' : str_replace(['🟦','🟩','🟥'], ['🏰','🏰','🏯'], $color);
            }

            // ژنرال‌ها
            if (!empty($tileGenerals)) {
                $genIcons = "";
                foreach ($tileGenerals as $gen) {
                    if ($gen['user_id'] == $user_id) {
                        $genIcons .= '💂‍♂️'; 
                    } elseif (checkAlliance($user_id, $gen['user_id'])) {
                        $genIcons .= '🛡️'; 
                    } else {
                        $genIcons .= '👹'; 
                    }
                }
                $icon = $genIcons;
            }
            
            if ($isPending) {
                $icon = "🎯" . $icon; 
            }
            
            $row_btns[] = ['text' => $icon, 'callback_data' => "tile_{$x}_{$y}"];
        }
        $keyboard[] = $row_btns;
    }
    
    // دکمه‌های جهت‌نما
    $jump_x = 3; $jump_y = 6;
    $keyboard[] = [
        ['text' => '⬅️', 'callback_data' => "map_".($center_x-$jump_x)."_$center_y"],
        ['text' => '⬆️', 'callback_data' => "map_$center_x"."_".($center_y-$jump_y)],
        ['text' => '⬇️', 'callback_data' => "map_$center_x"."_".($center_y+$jump_y)],
        ['text' => '➡️', 'callback_data' => "map_".($center_x+$jump_x)."_$center_y"]
    ];

    // *** (جدید) دکمه بازگشت به پایتخت ***
    // فقط اگر مرکز نقشه فعلی روی پایتخت نباشد نشان داده می‌شود
    $p = $conn->query("SELECT capital_x, capital_y FROM players WHERE user_id=$user_id")->fetch_assoc();
    if ($center_x != $p['capital_x'] || $center_y != $p['capital_y']) {
        $keyboard[] = [['text' => '🏠 بازگشت به پایتخت', 'callback_data' => "map_{$p['capital_x']}_{$p['capital_y']}"]];
    }
    
    sendMessage($chat_id, $map_display, ['inline_keyboard' => $keyboard]);
}

// ... (بقیه توابع handleTileSelect, handleGeneralSelect و غیره دقیقاً مثل قبل هستند) ...
// برای اطمینان، کدهای قبلی را اینجا تکرار می‌کنم که فایل کامل باشد.

function handleTileSelect($user_id, $chat_id, $tx, $ty) {
    global $conn;
    $sql = "SELECT t.*, p.username FROM map_tiles t LEFT JOIN players p ON t.owner_id = p.user_id WHERE x=$tx AND y=$ty";
    $tile = $conn->query($sql)->fetch_assoc();
    $genSql = "SELECT g.*, p.username FROM generals g JOIN players p ON g.user_id = p.user_id WHERE x=$tx AND y=$ty";
    $genResult = $conn->query($genSql);

    $buttons = [];
    $buttons[] = [['text' => '⚔️ اعزام ارتش', 'callback_data' => "ask_move_{$tx}_{$ty}"]];
    
    $infoText = "📍 **مختصات ($tx, $ty)**\n";
    if ($tile && $tile['owner_id']) {
        $ownerName = $tile['username'] ?: "بازیکن " . $tile['owner_id'];
        $infoText .= "🏳️ **قلمرو:** $ownerName\n";
        if ($tile['owner_id'] != $user_id) {
            $target_id = $tile['owner_id'];
            $buttons[] = [['text' => '🕵️ جاسوسی', 'callback_data' => "spy_ask_{$tx}_{$ty}"]];
            if (checkAlliance($user_id, $target_id)) {
                $buttons[] = [['text' => "💔 لغو اتحاد با $ownerName", 'callback_data' => "dip_break_$target_id"]];
            } else {
                $buttons[] = [['text' => "🤝 پیشنهاد اتحاد به $ownerName", 'callback_data' => "dip_req_$target_id"]];
            }
        }
    } else {
        $infoText .= "🏳️ **قلمرو:** بی‌طرف\n";
    }
    if ($genResult->num_rows > 0) {
        $infoText .= "\n🛡 **ارتش‌های حاضر:**\n";
        while ($gen = $genResult->fetch_assoc()) {
            $genOwnerName = $gen['username'] ?: "بازیکن " . $gen['user_id'];
            $relation = ($gen['user_id'] == $user_id) ? "(خودی)" : (checkAlliance($user_id, $gen['user_id']) ? "(متحد)" : "(دشمن)");
            $icon = ($gen['user_id'] == $user_id) ? '💂‍♂️' : (checkAlliance($user_id, $gen['user_id']) ? '🛡️' : '👹');
            $infoText .= "$icon **{$gen['name']}** ($genOwnerName) $relation\n";
        }
    }
    sendMessage($chat_id, $infoText, ['inline_keyboard' => $buttons]);
}

function handleGeneralSelect($user_id, $chat_id, $target_x, $target_y) {
    global $conn;
    $sql = "SELECT * FROM generals WHERE user_id = $user_id";
    $result = $conn->query($sql);
    if ($result->num_rows == 0) { sendMessage($chat_id, "❌ شما هیچ ژنرالی ندارید!"); return; }
    $keyboard = [];
    while ($gen = $result->fetch_assoc()) {
        $text = "🎖 {$gen['name']} (MP: {$gen['mp']}) [{$gen['x']},{$gen['y']}]";
        $keyboard[] = [['text' => $text, 'callback_data' => "exec_move_{$gen['id']}_{$target_x}_{$target_y}"]];
    }
    $keyboard[] = [['text' => '🔙 لغو', 'callback_data' => "map_{$target_x}_{$target_y}"]];
    sendMessage($chat_id, "📍 مقصد: ($target_x, $target_y)\nکدام ژنرال را اعزام می‌کنید؟", ['inline_keyboard' => $keyboard]);
}

function handleMoveOrder($user_id, $general_id, $target_x, $target_y) {
    global $conn;
    $gen = $conn->query("SELECT * FROM generals WHERE id = $general_id AND user_id = $user_id")->fetch_assoc();
    if (!$gen) return "❌ خطا: ژنرال یافت نشد.";
    $checkOrder = $conn->query("SELECT * FROM turn_orders WHERE general_id = $general_id");
    $msg_prefix = "";
    if ($checkOrder->num_rows > 0) {
        $oldOrder = $checkOrder->fetch_assoc();
        $old_dist = calculateDistance($gen['x'], $gen['y'], $oldOrder['target_x'], $oldOrder['target_y']);
        $oldTile = $conn->query("SELECT type FROM map_tiles WHERE x={$oldOrder['target_x']} AND y={$oldOrder['target_y']}")->fetch_assoc();
        $oldCost = $old_dist * getTerrainCost($oldTile['type'] ?? 'plain');
        $conn->query("UPDATE generals SET mp = mp + $oldCost WHERE id = $general_id");
        $conn->query("DELETE FROM turn_orders WHERE id = {$oldOrder['id']}");
        $gen['mp'] += $oldCost;
        $msg_prefix = "🔄 دستور قبلی لغو شد (+$oldCost MP).\n";
    }
    $dist = calculateDistance($gen['x'], $gen['y'], $target_x, $target_y);
    $tileRes = $conn->query("SELECT type FROM map_tiles WHERE x=$target_x AND y=$target_y")->fetch_assoc();
    $cost = $dist * getTerrainCost($tileRes['type'] ?? 'plain');
    if ($cost > $gen['mp']) return $msg_prefix . "❌ انرژی کافی نیست!\nنیاز: $cost | دارید: {$gen['mp']}";
    $conn->query("UPDATE generals SET mp = mp - $cost WHERE id = $general_id");
    $stmt = $conn->prepare("INSERT INTO turn_orders (user_id, general_id, target_x, target_y, type) VALUES (?, ?, ?, ?, 'move')");
    $stmt->bind_param("iiii", $user_id, $general_id, $target_x, $target_y);
    $stmt->execute();
    return $msg_prefix . "✅ دستور حرکت به ($target_x, $target_y) ثبت شد.\nهزینه مصرف شده: $cost MP";
}

// ---------------------------------------------------------
// 2. توابع مدیریت ارتش (Army Management)
// ---------------------------------------------------------

function handleArmyMenu($user_id, $chat_id) {
    global $conn;
    $pRes = $conn->query("SELECT * FROM players WHERE user_id=$user_id")->fetch_assoc();
    $gens = $conn->query("SELECT * FROM generals WHERE user_id=$user_id");
    
    $txt = "⚔️ **ستاد فرماندهی**\nژنرال‌ها: " . $gens->num_rows . " / " . $pRes['general_limit'] . "\n\n👇 عملیات:";
    $keyboard = [];
    while ($gen = $gens->fetch_assoc()) {
        $status = ($gen['mp'] < $gen['max_mp']) ? "Running 🏃" : "Ready ✅";
        $keyboard[] = [['text' => "🎖 {$gen['name']} ($status)", 'callback_data' => "gen_view_{$gen['id']}"]];
    }
    $keyboard[] = [['text' => "➕ استخدام (500 طلا)", 'callback_data' => "gen_recruit"]];
    $keyboard[] = [['text' => "🏋️ آموزش سرباز (در پایتخت)", 'callback_data' => "train_menu"]];

    sendMessage($chat_id, $txt, ['inline_keyboard' => $keyboard]);
}

function handleGeneralProfile($user_id, $chat_id, $gen_id) {
    global $conn;
    $gen = $conn->query("SELECT * FROM generals WHERE id=$gen_id AND user_id=$user_id")->fetch_assoc();
    if (!$gen) { sendMessage($chat_id, "ژنرال یافت نشد."); return; }

    $troops = json_decode($gen['troops'], true);
    $troopTxt = empty($troops) ? "هیچ" : "";
    foreach ($troops as $name => $count) {
        if($count > 0) $troopTxt .= "\n▫️ ". (UNIT_STATS[$name]['name'] ?? $name) . ": $count";
    }

    $txt = "🎖 **{$gen['name']}**\n📍 ({$gen['x']}, {$gen['y']})\n⚡ MP: {$gen['mp']}\n\n💂‍♂️ ارتش:$troopTxt";

    $keyboard = [];
    $p = $conn->query("SELECT capital_x, capital_y FROM players WHERE user_id=$user_id")->fetch_assoc();
    
    if ($gen['x'] == $p['capital_x'] && $gen['y'] == $p['capital_y']) {
        $keyboard[] = [
            ['text' => "⬅️ برداشت از شهر", 'callback_data' => "trans_sel_load_{$gen['id']}"],
            ['text' => "➡️ واریز به شهر", 'callback_data' => "trans_sel_unload_{$gen['id']}"]
        ];
    } else {
        $txt .= "\n\n⚠️ _برای انتقال نیرو، به پایتخت برگردید._";
    }
    
    $keyboard[] = [['text' => "🔙 بازگشت", 'callback_data' => "army_menu"]];
    sendMessage($chat_id, $txt, ['inline_keyboard' => $keyboard]);
}

function handleRecruitGeneral($user_id, $chat_id) {
    global $conn;
    $res = $conn->query("SELECT gold FROM player_resources WHERE user_id=$user_id")->fetch_assoc();
    if ($res['gold'] < GENERAL_RECRUIT_COST) return "❌ طلا کافی نیست!";
    
    $conn->query("UPDATE player_resources SET gold = gold - ".GENERAL_RECRUIT_COST." WHERE user_id=$user_id");
    
    $p = $conn->query("SELECT capital_x, capital_y FROM players WHERE user_id=$user_id")->fetch_assoc();
    $name = "ژنرال ".rand(100,999);
    $troops = json_encode([]); $mp = BASE_MOVEMENT_POINTS;
    
    $stmt = $conn->prepare("INSERT INTO generals (user_id, name, x, y, mp, max_mp, troops) VALUES (?, ?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("isiiiis", $user_id, $name, $p['capital_x'], $p['capital_y'], $mp, $mp, $troops);
    $stmt->execute();
    
    return "✅ $name با موفقیت استخدام شد.";
}


// ---------------------------------------------------------
// 3. توابع آموزش سرباز (Training)
// ---------------------------------------------------------

function handleTrainMenu($user_id, $chat_id) {
    global $conn;
    $res = $conn->query("SELECT * FROM player_resources WHERE user_id=$user_id")->fetch_assoc();
    $txt = "🏋️ **پادگان آموزشی**\n💰 {$res['gold']} | 🪵 {$res['wood']} | 🌾 {$res['food']}\n\nهزینه برای ۱۰ واحد:";
    $keyboard = [];
    foreach (UNIT_STATS as $key => $stat) {
        $btnText = "{$stat['name']} (💰".($stat['gold']*10)." 🌾".($stat['food']*10).")";
        $keyboard[] = [['text' => $btnText, 'callback_data' => "train_do_{$key}"]];
    }
    $keyboard[] = [['text' => "🔙 بازگشت", 'callback_data' => "army_menu"]];
    sendMessage($chat_id, $txt, ['inline_keyboard' => $keyboard]);
}

function handleTrainAction($user_id, $unit_type) {
    global $conn;
    if (!isset(UNIT_STATS[$unit_type])) return "❌ واحد نامعتبر.";
    
    $count = 10; $cost = UNIT_STATS[$unit_type];
    $totalGold = $cost['gold']*$count; $totalFood = $cost['food']*$count;
    
    $res = $conn->query("SELECT * FROM player_resources WHERE user_id=$user_id")->fetch_assoc();
    if ($res['gold'] < $totalGold || $res['food'] < $totalFood) return "❌ منابع کافی نیست.";
    
    $conn->query("UPDATE player_resources SET gold=gold-$totalGold, food=food-$totalFood WHERE user_id=$user_id");
    
    $p = $conn->query("SELECT capital_x, capital_y FROM players WHERE user_id=$user_id")->fetch_assoc();
    $tile = $conn->query("SELECT garrison FROM map_tiles WHERE x={$p['capital_x']} AND y={$p['capital_y']}")->fetch_assoc();
    
    $garrison = json_decode($tile['garrison'], true) ?? [];
    if (!isset($garrison[$unit_type])) $garrison[$unit_type] = 0;
    $garrison[$unit_type] += $count;
    
    $conn->query("UPDATE map_tiles SET garrison='".json_encode($garrison)."' WHERE x={$p['capital_x']} AND y={$p['capital_y']}");
    
    return "✅ $count {$cost['name']} آموزش دید و به پادگان پایتخت اضافه شد.";
}


// ---------------------------------------------------------
// 4. سیستم پیشرفته انتقال نیرو (Transfer System)
// ---------------------------------------------------------

function handleTransferMenu($user_id, $chat_id, $gen_id, $direction) {
    global $conn;
    $gen = $conn->query("SELECT * FROM generals WHERE id=$gen_id")->fetch_assoc();
    $p = $conn->query("SELECT capital_x, capital_y FROM players WHERE user_id=$user_id")->fetch_assoc();
    
    if ($gen['x'] != $p['capital_x'] || $gen['y'] != $p['capital_y']) {
        sendMessage($chat_id, "❌ ژنرال در پایتخت نیست."); return;
    }
    
    $cityTile = $conn->query("SELECT garrison FROM map_tiles WHERE x={$p['capital_x']} AND y={$p['capital_y']}")->fetch_assoc();
    $cityTroops = json_decode($cityTile['garrison'], true) ?? [];
    $genTroops = json_decode($gen['troops'], true) ?? [];
    
    $sourceTroops = ($direction == 'load') ? $cityTroops : $genTroops;
    $actionName = ($direction == 'load') ? "برداشت از شهر" : "واریز به شهر";
    
    if (empty($sourceTroops)) {
        sendMessage($chat_id, "❌ هیچ نیرویی برای انتقال وجود ندارد."); return;
    }
    
    $txt = "🔄 **انتقال نیرو ($actionName)**\nکدام واحد را منتقل می‌کنید؟";
    $keyboard = [];
    
    foreach ($sourceTroops as $type => $count) {
        if ($count <= 0) continue;
        $uName = UNIT_STATS[$type]['name'] ?? $type;
        $keyboard[] = [['text' => "$uName (موجود: $count)", 'callback_data' => "trans_qty_{$direction}_{$gen_id}_{$type}"]];
    }
    $keyboard[] = [['text' => "🔙 بازگشت", 'callback_data' => "gen_view_{$gen_id}"]];
    
    sendMessage($chat_id, $txt, ['inline_keyboard' => $keyboard]);
}

function handleTransferAmount($user_id, $chat_id, $gen_id, $direction, $unit_type) {
    global $conn;
    $gen = $conn->query("SELECT * FROM generals WHERE id=$gen_id")->fetch_assoc();
    $p = $conn->query("SELECT capital_x, capital_y FROM players WHERE user_id=$user_id")->fetch_assoc();
    
    $cityTile = $conn->query("SELECT garrison FROM map_tiles WHERE x={$p['capital_x']} AND y={$p['capital_y']}")->fetch_assoc();
    $cityTroops = json_decode($cityTile['garrison'], true) ?? [];
    $genTroops = json_decode($gen['troops'], true) ?? [];
    
    $max = ($direction == 'load') ? ($cityTroops[$unit_type] ?? 0) : ($genTroops[$unit_type] ?? 0);
    $uName = UNIT_STATS[$unit_type]['name'] ?? $unit_type;
    $act = ($direction == 'load') ? "به ژنرال" : "به شهر";
    
    $txt = "🔢 **تعداد انتقال $uName $act:**\nحداکثر موجودی: $max";
    
    $keyboard = [];
    $amounts = [10, 50, 100];
    $row = [];
    foreach($amounts as $amt) {
        if ($amt <= $max) {
            $row[] = ['text' => "$amt", 'callback_data' => "trans_do_{$direction}_{$gen_id}_{$unit_type}_{$amt}"];
        }
    }
    if (!empty($row)) $keyboard[] = $row;
    
    $keyboard[] = [['text' => "همه ($max)", 'callback_data' => "trans_do_{$direction}_{$gen_id}_{$unit_type}_max"]];
    $keyboard[] = [['text' => "🔙 بازگشت", 'callback_data' => "trans_sel_{$direction}_{$gen_id}"]];
    
    sendMessage($chat_id, $txt, ['inline_keyboard' => $keyboard]);
}

function handleTransferExecute($user_id, $gen_id, $direction, $unit_type, $amount_req) {
    global $conn;
    
    $gen = $conn->query("SELECT * FROM generals WHERE id=$gen_id AND user_id=$user_id")->fetch_assoc();
    $p = $conn->query("SELECT capital_x, capital_y FROM players WHERE user_id=$user_id")->fetch_assoc();
    
    if ($gen['x'] != $p['capital_x'] || $gen['y'] != $p['capital_y']) return "❌ خطا: موقعیت نامعتبر.";
    
    $cityTile = $conn->query("SELECT garrison FROM map_tiles WHERE x={$p['capital_x']} AND y={$p['capital_y']}")->fetch_assoc();
    $cityTroops = json_decode($cityTile['garrison'], true) ?? [];
    $genTroops = json_decode($gen['troops'], true) ?? [];
    
    if ($direction == 'load') { 
        $max = $cityTroops[$unit_type] ?? 0;
    } else { 
        $max = $genTroops[$unit_type] ?? 0;
    }
    
    $amount = ($amount_req == 'max') ? $max : intval($amount_req);
    if ($amount > $max || $amount <= 0) return "❌ تعداد نامعتبر.";
    
    if ($direction == 'load') {
        $cityTroops[$unit_type] -= $amount;
        if (!isset($genTroops[$unit_type])) $genTroops[$unit_type] = 0;
        $genTroops[$unit_type] += $amount;
    } else {
        $genTroops[$unit_type] -= $amount;
        if (!isset($cityTroops[$unit_type])) $cityTroops[$unit_type] = 0;
        $cityTroops[$unit_type] += $amount;
    }
    
    $cityJson = json_encode($cityTroops);
    $genJson = json_encode($genTroops);
    
    $conn->query("UPDATE map_tiles SET garrison='$cityJson' WHERE x={$p['capital_x']} AND y={$p['capital_y']}");
    $conn->query("UPDATE generals SET troops='$genJson' WHERE id=$gen_id");
    
    $uName = UNIT_STATS[$unit_type]['name'] ?? $unit_type;
    return "✅ انتقال $amount $uName با موفقیت انجام شد.";
}


// ---------------------------------------------------------
// 5. مدیریت شهر و ساختمان‌ها (City)
// ---------------------------------------------------------

function handleCityMenu($user_id, $chat_id) {
    global $conn;
    
    $bResult = $conn->query("SELECT * FROM buildings WHERE user_id=$user_id");
    $levels = [];
    while($row = $bResult->fetch_assoc()) $levels[$row['type']] = $row['level'];
    
    $qCheck = $conn->query("SELECT * FROM building_queue WHERE user_id=$user_id LIMIT 1");
    $busy = $qCheck->fetch_assoc();
    
    $status = $busy ? "🚧 کارگران مشغول ساخت [{$busy['building']}] هستند." : "✅ کارگران آماده کار هستند.";
    $txt = "🏰 **مدیریت شهر**\n\n$status\n\n👇 ساختمان مورد نظر را انتخاب کنید:";
    
    $keyboard = [];
    foreach (BUILDING_DATA as $type => $data) {
        $lvl = $levels[$type] ?? 0; 
        $keyboard[] = [['text' => "{$data['name']} (سطح $lvl)", 'callback_data' => "build_view_$type"]];
    }
    $keyboard[] = [['text' => "🔙 بازگشت", 'callback_data' => "army_menu"]];
    
    sendMessage($chat_id, $txt, ['inline_keyboard' => $keyboard]);
}

function handleBuildView($user_id, $chat_id, $type) {
    global $conn;
    if (!isset(BUILDING_DATA[$type])) return;
    
    $bRes = $conn->query("SELECT level FROM buildings WHERE user_id=$user_id AND type='$type'");
    $row = $bRes->fetch_assoc();
    $currentLvl = $row ? $row['level'] : 0;
    $nextLvl = $currentLvl + 1;
    
    $data = BUILDING_DATA[$type];
    $multiplier = pow(BUILD_COST_MULTIPLIER, $currentLvl);
    $woodCost = floor($data['base_cost']['wood'] * $multiplier);
    $stoneCost = floor($data['base_cost']['stone'] * $multiplier);
    $timeMin = floor($data['time_min'] * $multiplier);
    
    $txt = "🏗 **{$data['name']}**\n📝 {$data['desc']}\n\n🔹 سطح فعلی: $currentLvl\n🔸 سطح بعدی: $nextLvl\n\n💰 هزینه:\n🪵 چوب: $woodCost\n🪨 سنگ: $stoneCost\n⏳ زمان: $timeMin دقیقه";
    
    $keyboard = [
        [['text' => "🔨 ارتقا به سطح $nextLvl", 'callback_data' => "build_start_$type"]],
        [['text' => "🔙 بازگشت", 'callback_data' => "city_menu"]]
    ];
    sendMessage($chat_id, $txt, ['inline_keyboard' => $keyboard]);
}

function handleBuildStart($user_id, $type) {
    global $conn;
    $qCheck = $conn->query("SELECT id FROM building_queue WHERE user_id=$user_id");
    if ($qCheck->num_rows > 0) return "❌ کارگران مشغول هستند!";
    
    $bRes = $conn->query("SELECT level FROM buildings WHERE user_id=$user_id AND type='$type'");
    $row = $bRes->fetch_assoc();
    $currentLvl = $row ? $row['level'] : 0;
    $nextLvl = $currentLvl + 1;
    
    $data = BUILDING_DATA[$type];
    $multiplier = pow(BUILD_COST_MULTIPLIER, $currentLvl);
    $woodCost = floor($data['base_cost']['wood'] * $multiplier);
    $stoneCost = floor($data['base_cost']['stone'] * $multiplier);
    $timeMin = floor($data['time_min'] * $multiplier);
    
    $res = $conn->query("SELECT wood, stone FROM player_resources WHERE user_id=$user_id")->fetch_assoc();
    if ($res['wood'] < $woodCost || $res['stone'] < $stoneCost) return "❌ منابع کافی نیست!";
    
    $conn->query("UPDATE player_resources SET wood = wood - $woodCost, stone = stone - $stoneCost WHERE user_id=$user_id");
    
    $finishTime = date('Y-m-d H:i:s', strtotime("+$timeMin minutes"));
    $stmt = $conn->prepare("INSERT INTO building_queue (user_id, building, level, finish_time) VALUES (?, ?, ?, ?)");
    $stmt->bind_param("isis", $user_id, $type, $nextLvl, $finishTime);
    $stmt->execute();
    
    return "✅ ساخت آغاز شد. ($timeMin دقیقه)";
}


// ---------------------------------------------------------
// 6. سیستم جاسوسی (Espionage)
// ---------------------------------------------------------

function handleSpyMenu($user_id, $chat_id, $target_x, $target_y) {
    global $conn;
    $p = $conn->query("SELECT spy_status, spy_cooldown FROM players WHERE user_id=$user_id")->fetch_assoc();
    
    // خودترمیمی: اگر وضعیت on_mission است ولی رکوردی ندارد
    if ($p['spy_status'] == 'on_mission') {
        $check = $conn->query("SELECT mission_id FROM espionage_missions WHERE user_id=$user_id");
        if ($check->num_rows == 0) {
            $conn->query("UPDATE players SET spy_status='ready' WHERE user_id=$user_id");
            $p['spy_status'] = 'ready';
        } else {
            return sendMessage($chat_id, "❌ جاسوس در مأموریت است.");
        }
    }

    if ($p['spy_status'] == 'recovering') {
        $date = new DateTime($p['spy_cooldown']);
        $now = new DateTime();
        if ($date > $now) {
            $diff = $date->diff($now);
            return sendMessage($chat_id, "❌ جاسوس غیرفعال است.\n⏳ {$diff->h} ساعت دیگر برمی‌گردد.");
        } else {
            $conn->query("UPDATE players SET spy_status='ready', spy_cooldown=NULL WHERE user_id=$user_id");
        }
    }
    
    $txt = "🕵️ **عملیات سری**\nهدف: ($target_x, $target_y)";
    $keyboard = [
        [['text' => '💰 بررسی منابع (100 طلا)', 'callback_data' => "spy_do_resources_{$target_x}_{$target_y}"]],
        [['text' => '💂‍♂️ شمارش نیرو (100 طلا)', 'callback_data' => "spy_do_troops_{$target_x}_{$target_y}"]],
        [['text' => '🔙 لغو', 'callback_data' => "map_{$target_x}_{$target_y}"]]
    ];
    sendMessage($chat_id, $txt, ['inline_keyboard' => $keyboard]);
}

function handleSpyOrder($user_id, $type, $tx, $ty) {
    global $conn;
    $tile = $conn->query("SELECT owner_id FROM map_tiles WHERE x=$tx AND y=$ty")->fetch_assoc();
    if (!$tile || !$tile['owner_id']) return "❌ هدف نامعتبر.";
    if ($tile['owner_id'] == $user_id) return "❌ جاسوسی از خود؟";
    
    $cost = 100;
    $res = $conn->query("SELECT gold FROM player_resources WHERE user_id=$user_id")->fetch_assoc();
    if ($res['gold'] < $cost) return "❌ طلا کافی نیست.";
    
    $conn->query("UPDATE player_resources SET gold = gold - $cost WHERE user_id=$user_id");
    $conn->query("UPDATE players SET spy_status = 'on_mission' WHERE user_id=$user_id");
    
    $mType = ($type == 'resources') ? 'check_resources' : 'check_troops';
    $stmt = $conn->prepare("INSERT INTO espionage_missions (user_id, target_user_id, mission_type) VALUES (?, ?, ?)");
    $stmt->bind_param("iis", $user_id, $tile['owner_id'], $mType);
    $stmt->execute();
    
    return "✅ جاسوس اعزام شد.\nگزارش در پایان نوبت ارسال می‌شود.";
}

// ---------------------------------------------------------
// 7. سیستم رتبه‌بندی (Ranking)
// ---------------------------------------------------------

function handleRanking($user_id, $chat_id) {
    global $conn;
    
    $sql = "SELECT p.user_id, p.username,
            (
                (SELECT COUNT(*) FROM generals WHERE user_id=p.user_id) * 50 +
                (SELECT COUNT(*) FROM map_tiles WHERE owner_id=p.user_id AND type='capital') * 500 +
                (SELECT COALESCE(SUM(level), 0) FROM buildings WHERE user_id=p.user_id) * 10
            ) as base_score
            FROM players p";
            
    $result = $conn->query($sql);
    $scores = [];
    
    while ($row = $result->fetch_assoc()) {
        $uid = $row['user_id'];
        $score = $row['base_score'];
        
        $troopsCount = 0;
        $genRes = $conn->query("SELECT troops FROM generals WHERE user_id=$uid");
        while($g = $genRes->fetch_assoc()) {
            $arr = json_decode($g['troops'], true) ?? [];
            foreach($arr as $c) $troopsCount += $c;
        }
        $tileRes = $conn->query("SELECT garrison FROM map_tiles WHERE owner_id=$uid AND type='capital'");
        while($t = $tileRes->fetch_assoc()) {
            $arr = json_decode($t['garrison'], true) ?? [];
            foreach($arr as $c) $troopsCount += $c;
        }
        
        $score += $troopsCount;
        $scores[] = ['uid' => $uid, 'name' => $row['username'], 'score' => $score];
    }
    
    usort($scores, function($a, $b) {
        return $b['score'] - $a['score'];
    });
    
    $txt = "🏆 **برترین فرماندهان**\n\n";
    $rank = 1;
    $userRank = "ثبت نشده";
    
    foreach ($scores as $s) {
        if ($rank <= 10) {
            $medal = ($rank == 1) ? "🥇" : (($rank == 2) ? "🥈" : (($rank == 3) ? "🥉" : "🎗"));
            $name = $s['name'] ?: "بازیکن $rank";
            $txt .= "$medal **{$rank}.** {$name} \n   └ 🎖 امتیاز: " . number_format($s['score']) . "\n";
        }
        if ($s['uid'] == $user_id) {
            $userRank = $rank;
        }
        $rank++;
    }
    
    $txt .= "\n📊 **رتبه شما:** $userRank";
    
    $keyboard = [[['text' => '🔄 بروزرسانی', 'callback_data' => 'show_ranking']]];
    sendMessage($chat_id, $txt, ['inline_keyboard' => $keyboard]);
}

// ---------------------------------------------------------
// 8. سیستم دیپلماسی (Diplomacy)
// ---------------------------------------------------------

function handleDiplomacyMenu($user_id, $chat_id) {
    global $conn;
    
    $txt = "🤝 **وزارت امور خارجه**\n\n";
    
    $reqs = $conn->query("SELECT d.id, p.username, p.user_id FROM diplomacy d JOIN players p ON d.requester_id = p.user_id WHERE d.receiver_id = $user_id AND d.status = 'pending'");
    
    $keyboard = [];
    if ($reqs->num_rows > 0) {
        $txt .= "📩 **درخواست‌های جدید:**\n";
        while($row = $reqs->fetch_assoc()) {
            $name = $row['username'] ?: "بازیکن {$row['user_id']}";
            $txt .= "▫️ از طرف: {$name}\n";
            $keyboard[] = [
                ['text' => "✅ قبول", 'callback_data' => "dip_acc_{$row['id']}"],
                ['text' => "❌ رد", 'callback_data' => "dip_rej_{$row['id']}"]
            ];
        }
    } else {
        $txt .= "📩 صندوق ورودی خالی است.\n";
    }
    
    $allies = $conn->query("SELECT p.username, p.user_id FROM diplomacy d 
                            JOIN players p ON (d.requester_id = p.user_id OR d.receiver_id = p.user_id)
                            WHERE (d.requester_id = $user_id OR d.receiver_id = $user_id) 
                            AND d.status = 'ally' AND p.user_id != $user_id");
                            
    $txt .= "\n🛡 **متحدین شما:**\n";
    if ($allies->num_rows > 0) {
        while($a = $allies->fetch_assoc()) {
            $name = $a['username'] ?: "بازیکن {$a['user_id']}";
            $txt .= "🟩 {$name}\n";
        }
    } else {
        $txt .= "هیچ متحدی ندارید.\n";
    }
    
    $keyboard[] = [['text' => "🔙 بازگشت", 'callback_data' => "army_menu"]];
    sendMessage($chat_id, $txt, ['inline_keyboard' => $keyboard]);
}

function handleSendDiplomacyRequest($user_id, $target_id) {
    global $conn;
    $check = $conn->query("SELECT id FROM diplomacy WHERE (requester_id=$user_id AND receiver_id=$target_id) OR (requester_id=$target_id AND receiver_id=$user_id)");
    if ($check->num_rows > 0) return "⚠️ درخواست تکراری یا اتحاد موجود.";
    
    $stmt = $conn->prepare("INSERT INTO diplomacy (requester_id, receiver_id, status) VALUES (?, ?, 'pending')");
    $stmt->bind_param("ii", $user_id, $target_id);
    $stmt->execute();
    
    sendMessage($target_id, "📩 **پیشنهاد اتحاد!**\nبازیکنی به شما پیشنهاد اتحاد داد. به منوی دیپلماسی بروید.");
    return "✅ پیشنهاد ارسال شد.";
}

function handleDiplomacyAction($user_id, $req_id, $action) {
    global $conn;
    $req = $conn->query("SELECT * FROM diplomacy WHERE id=$req_id AND receiver_id=$user_id AND status='pending'")->fetch_assoc();
    if (!$req) return "❌ درخواست نامعتبر.";
    
    if ($action == 'accept') {
        $conn->query("UPDATE diplomacy SET status='ally' WHERE id=$req_id");
        sendMessage($req['requester_id'], "🤝 **تبریک!** اتحاد برقرار شد.");
        return "✅ اکنون متحد هستید.";
    } else {
        $conn->query("DELETE FROM diplomacy WHERE id=$req_id");
        sendMessage($req['requester_id'], "❌ پیشنهاد اتحاد رد شد.");
        return "❌ درخواست رد شد.";
    }
}

function handleBreakAlliance($user_id, $target_id) {
    global $conn;
    $conn->query("DELETE FROM diplomacy WHERE (requester_id=$user_id AND receiver_id=$target_id) OR (requester_id=$target_id AND receiver_id=$user_id)");
    sendMessage($target_id, "💔 **هشدار!** پیمان اتحاد شکسته شد.");
    return "💔 اتحاد لغو شد.";
}
?>