<?php
// Filename: includes/turn_engine.php
// Description: موتور پردازش نوبت (جنگ، اقتصاد، جاسوسی، رویدادها)

require_once __DIR__ . '/db_connect.php';
require_once __DIR__ . '/functions.php';

// تابع اصلی که توسط Cron Job یا ادمین صدا زده می‌شود
function executeTurn() {
    global $conn;
    $logs = [];
    $logs[] = "⏳ شروع پردازش نوبت...";

    // 1. پردازش جاسوسی (اطلاعات قبل از جنگ به دست می‌آید)
    processEspionage(); 
    $logs[] = "🕵️ عملیات جاسوسی انجام شد.";

    // 2. دریافت دستورات حرکت و نبرد
    $sql = "SELECT o.*, g.troops, g.user_id as owner_id, g.name as gen_name, g.id as general_id 
            FROM turn_orders o 
            JOIN generals g ON o.general_id = g.id 
            ORDER BY o.created_at ASC";
    $result = $conn->query($sql);

    $movements = [];
    // گروه‌بندی بر اساس مختصات مقصد
    while ($row = $result->fetch_assoc()) {
        $key = $row['target_x'] . '_' . $row['target_y'];
        $movements[$key][] = $row;
    }

    // پردازش هر مقصد
    foreach ($movements as $coord => $armies) {
        list($tx, $ty) = explode('_', $coord);
        
        // گروه‌بندی ارتش‌ها بر اساس مالک (برای حمله ترکیبی)
        $playerGroups = [];
        foreach ($armies as $army) {
            $playerGroups[$army['owner_id']][] = $army;
        }

        // پردازش ورود هر گروه به خانه
        foreach ($playerGroups as $owner_id => $generals) {
             processGroupArrival($generals, $tx, $ty);
        }
    }
    $logs[] = "✅ نبردها و حرکت‌ها پردازش شد.";

    // پاکسازی دستورات و ریست انرژی ژنرال‌ها
    $conn->query("TRUNCATE TABLE turn_orders");
    $conn->query("UPDATE generals SET mp = max_mp"); 
    
    // 3. فاز اقتصادی (تولید منابع + مصرف غذا + ساخت و ساز)
    processDailyReportAndUpkeep();
    
    // 4. بررسی وضعیت اضطراری (Last Stand)
    checkLastStandStatus();

    $logs[] = "🏁 پایان پردازش نوبت.";
    return implode("\n", $logs);
}

// ------------------------------------------------------------------
// تابع پردازش ورود یک گروه ارتش (متعلق به یک نفر) به یک خانه
// ------------------------------------------------------------------
function processGroupArrival($generals, $x, $y) {
    global $conn;
    $owner_id = $generals[0]['owner_id'];
    $ownerName = getPlayerName($owner_id);
    
    // 1. ادغام نیروها برای محاسبه قدرت کل
    $totalAttackers = [];
    foreach ($generals as $gen) {
        $troops = json_decode($gen['troops'], true) ?? [];
        foreach ($troops as $t => $c) {
            if (!isset($totalAttackers[$t])) $totalAttackers[$t] = 0;
            $totalAttackers[$t] += $c;
        }
    }

    // 2. شناسایی وضعیت مقصد
    // الف) آیا ژنرال دیگری (غیر از خودم) آنجاست؟
    $otherGenStmt = $conn->prepare("SELECT * FROM generals WHERE x = ? AND y = ? AND user_id != ? LIMIT 1");
    $otherGenStmt->bind_param("iii", $x, $y, $owner_id);
    $otherGenStmt->execute();
    $otherGen = $otherGenStmt->get_result()->fetch_assoc();

    $defenderTroops = [];
    $battleType = 'none'; // none, field, siege
    $tile = null;
    $target_owner_id = null;
    $isAllyAction = false; // پرچم عملیات دوستانه

    // ب) دریافت اطلاعات زمین
    $stmt = $conn->prepare("SELECT * FROM map_tiles WHERE x = ? AND y = ?");
    $stmt->bind_param("ii", $x, $y);
    $stmt->execute();
    $tile = $stmt->get_result()->fetch_assoc();

    // ج) تصمیم‌گیری بر اساس موجودات در مقصد
    if ($otherGen) {
        // یک ژنرال دیگر اینجاست
        if (checkAlliance($owner_id, $otherGen['user_id'])) {
            // متحد است -> حرکت مجاز (استقرار مشترک / Reinforce)
            $battleType = 'none';
            $isAllyAction = true;
            $target_owner_id = $otherGen['user_id'];
        } else {
            // دشمن است -> جنگ تن‌به‌تن (Field Battle)
            $battleType = 'field';
            $defenderTroops = json_decode($otherGen['troops'], true) ?? [];
            $target_owner_id = $otherGen['user_id'];
        }
    } elseif ($tile && $tile['type'] == 'capital' && $tile['owner_id'] && $tile['owner_id'] != $owner_id) {
        // قلعه‌ای وجود دارد (و ژنرالی رویش نیست)
        if (checkAlliance($owner_id, $tile['owner_id'])) {
            // قلعه متحد -> حرکت مجاز (بازدید/دفاع)
            $battleType = 'none';
            $isAllyAction = true;
            $target_owner_id = $tile['owner_id'];
        } else {
            // قلعه دشمن -> محاصره (Siege)
            $battleType = 'siege';
            $defenderTroops = json_decode($tile['garrison'], true) ?? [];
            $target_owner_id = $tile['owner_id'];
        }
    }

    // 3. انجام حرکت (بدون جنگ)
    if ($battleType == 'none') {
        // آپدیت مختصات تمام ژنرال‌های گروه
        foreach ($generals as $gen) {
            updateGeneralPos($gen['general_id'], $x, $y);
        }
        
        if ($isAllyAction) {
             // سناریوی کمک به متحد
             $allyName = getPlayerName($target_owner_id);
             sendMessage($owner_id, "🤝 **استقرار موفق!**\nنیروهای شما به موقعیت متحدتان ($allyName) رسیدند و در کنار آن‌ها مستقر شدند.");
             sendMessage($target_owner_id, "🛡️ **نیروی کمکی!**\nارتش متحد ($ownerName) برای پشتیبانی به موقعیت ($x, $y) رسید.");
        } 
        elseif ($tile && $tile['type'] == 'capital' && !$tile['owner_id']) {
             // تسخیر قلعه خالی
             $conn->query("INSERT INTO map_tiles (x, y, owner_id, type) VALUES ($x, $y, $owner_id, 'capital') ON DUPLICATE KEY UPDATE owner_id=$owner_id");
             sendMessage($owner_id, "🚩 ارتش شما قلعه‌ای بی‌طرف را در ($x, $y) تسخیر کرد!");
        } 
        else {
             // حرکت معمولی در دشت
             // sendMessage($owner_id, "✅ ارتش به ($x, $y) رسید.");
        }
        return;
    }

    // 4. انجام نبرد (فقط اگر دشمن باشد)
    $result = calculateBattle($totalAttackers, $defenderTroops);
    $targetName = getPlayerName($target_owner_id);
    
    // --- محاسبه تلفات مهاجم (و توزیع بین ژنرال‌ها) ---
    $attTotalLost = [];
    
    // استفاده از & برای تغییر مستقیم آرایه اصلی generals
    foreach ($generals as &$gen) {
        $originalTroops = json_decode($gen['troops'], true) ?? [];
        $newTroops = [];
        foreach ($originalTroops as $t => $c) {
            $survivors = floor($c * (1 - $result['att_loss_pct']));
            if ($survivors > 0) $newTroops[$t] = $survivors;
            
            $lost = $c - $survivors;
            if ($lost > 0) {
                if (!isset($attTotalLost[$t])) $attTotalLost[$t] = 0;
                $attTotalLost[$t] += $lost;
            }
        }
        $gen['new_troops'] = $newTroops; 
    }
    unset($gen); // پایان رفرنس

    // محاسبه تلفات مدافع برای گزارش
    $defTotalLost = [];
    foreach ($defenderTroops as $t => $c) {
        $surv = $result['defender_survivors'][$t] ?? 0;
        $lost = $c - $surv;
        if ($lost > 0) $defTotalLost[$t] = $lost;
    }

    $attReport = formatLosses($attTotalLost);
    $defReport = formatLosses($defTotalLost);
    
    // 5. اعمال نتایج نبرد
    if ($battleType == 'field') {
        // نبرد با ژنرال دشمن
        $defJson = json_encode($result['defender_survivors']);
        $conn->query("UPDATE generals SET troops='$defJson' WHERE id={$otherGen['id']}");
        
        // ذخیره نیروهای جدید مهاجم
        foreach ($generals as $gen) {
             $json = json_encode($gen['new_troops'] ?? []); 
             $conn->query("UPDATE generals SET troops='$json' WHERE id={$gen['general_id']}");
             // اگر مهاجم برد، پیشروی می‌کند
             if ($result['winner'] == 'attacker') updateGeneralPos($gen['general_id'], $x, $y);
        }

        if ($result['winner'] == 'attacker') {
            retreatGeneral($otherGen); // دشمن عقب‌نشینی می‌کند
            sendMessage($owner_id, "⚔️ **پیروزی بر $targetName!**\n💀 تلفات شما: $attReport\n💀 تلفات دشمن: $defReport");
            sendMessage($target_owner_id, "🩸 **شکست در برابر $ownerName!**\nارتش شما در ($x, $y) درهم شکست و عقب‌نشینی کرد.\n💀 تلفات شما: $defReport\n💀 تلفات دشمن: $attReport");
        } else {
            sendMessage($owner_id, "💀 **شکست در برابر $targetName!**\nحمله شما دفع شد.\n💀 تلفات شما: $attReport\n💀 تلفات دشمن: $defReport");
            sendMessage($target_owner_id, "🛡 **دفاع موفق!**\nحمله $ownerName دفع شد.\n💀 تلفات شما: $defReport\n💀 تلفات دشمن: $attReport");
        }

    } elseif ($battleType == 'siege') {
        // محاصره قلعه
        if ($result['winner'] == 'attacker') {
            $resLootMsg = "";
            
            // الف) غارت منابع
            $defRes = $conn->query("SELECT * FROM player_resources WHERE user_id=$target_owner_id")->fetch_assoc();
            if ($defRes) {
                $conn->query("UPDATE player_resources SET 
                    food = food + {$defRes['food']}, 
                    wood = wood + {$defRes['wood']}, 
                    stone = stone + {$defRes['stone']}, 
                    gold = gold + {$defRes['gold']} 
                    WHERE user_id=$owner_id");
                    
                $conn->query("UPDATE player_resources SET food=0, wood=0, stone=0, gold=0 WHERE user_id=$target_owner_id");
                
                $resLootMsg = "\n💰 **غنایم:** {$defRes['gold']} طلا...";
            }

            // ب) اسارت سربازان (Conscript)
            $conscriptMsg = "";
            $conscriptedTroops = [];
            foreach ($result['defender_survivors'] as $type => $count) {
                $amount = floor($count * 0.20); // 20% باقیمانده‌ها اسیر می‌شوند
                if ($amount > 0) $conscriptedTroops[$type] = $amount;
            }
            
            if (!empty($conscriptedTroops)) {
                // اضافه کردن به اولین ژنرال مهاجم
                $firstGen = &$generals[0];
                if (!isset($firstGen['new_troops'])) $firstGen['new_troops'] = [];
                
                foreach ($conscriptedTroops as $type => $count) {
                    if (!isset($firstGen['new_troops'][$type])) $firstGen['new_troops'][$type] = 0;
                    $firstGen['new_troops'][$type] += $count;
                }
                unset($firstGen);
                $conscriptMsg = "\n🔗 **اسرا:** " . formatLosses($conscriptedTroops) . " به ارتش پیوستند.";
            }

            // ذخیره نهایی مهاجم
            foreach ($generals as $gen) {
                $json = json_encode($gen['new_troops'] ?? []);
                $conn->query("UPDATE generals SET troops='$json' WHERE id={$gen['general_id']}");
                updateGeneralPos($gen['general_id'], $x, $y);
            }

            // تغییر مالکیت قلعه
            $conn->query("UPDATE map_tiles SET owner_id = $owner_id, garrison = NULL WHERE x=$x AND y=$y");
            
            sendMessage($owner_id, "⚔️ **قلعه $targetName فتح شد!**\n💀 تلفات شما: $attReport\n💀 تلفات دشمن: $defReport $resLootMsg $conscriptMsg");
            sendMessage($target_owner_id, "🔥 **قلعه سقوط کرد!**\nشهر شما توسط $ownerName غارت و تسخیر شد.\n💀 تلفات: $defReport");

        } else {
            // شکست مهاجم
            foreach ($generals as $gen) { 
                $json = json_encode($gen['new_troops'] ?? []);
                $conn->query("UPDATE generals SET troops='$json' WHERE id={$gen['general_id']}");
            }
            
            $newGarrison = json_encode($result['defender_survivors']);
            $conn->query("UPDATE map_tiles SET garrison = '$newGarrison' WHERE x=$x AND y=$y");
            
            sendMessage($owner_id, "💀 **شکست در محاصره!**\nحمله به قلعه $targetName ناکام ماند.\n💀 تلفات شما: $attReport");
            sendMessage($target_owner_id, "🛡 **مقاومت جانانه!**\nقلعه در برابر حمله $ownerName حفظ شد.\n💀 تلفات دشمن: $attReport");
        }
    }
}

// ------------------------------------------------------------------
// توابع کمکی جاسوسی و اقتصاد
// ------------------------------------------------------------------

function processEspionage() {
    global $conn;
    
    $sql = "SELECT * FROM espionage_missions";
    $result = $conn->query($sql);
    
    while ($mission = $result->fetch_assoc()) {
        $spyId = $mission['user_id'];
        $targetId = $mission['target_user_id'];
        $targetName = getPlayerName($targetId);
        
        $chance = ($mission['mission_type'] == 'check_resources') ? 80 : 50;
        $roll = rand(1, 100);
        
        if ($roll <= $chance) {
            $report = "🕵️ **گزارش جاسوسی از $targetName**\n";
            if ($mission['mission_type'] == 'check_resources') {
                $res = $conn->query("SELECT * FROM player_resources WHERE user_id=$targetId")->fetch_assoc();
                $report .= "خزانه:\n💰 {$res['gold']} | 🌾 {$res['food']} | 🪵 {$res['wood']} | 🪨 {$res['stone']}";
            } else {
                $p = $conn->query("SELECT capital_x, capital_y FROM players WHERE user_id=$targetId")->fetch_assoc();
                $tile = $conn->query("SELECT garrison FROM map_tiles WHERE x={$p['capital_x']} AND y={$p['capital_y']}")->fetch_assoc();
                $totalTroops = [];
                $garrison = json_decode($tile['garrison'], true) ?? [];
                foreach ($garrison as $k => $v) { if(!isset($totalTroops[$k])) $totalTroops[$k]=0; $totalTroops[$k]+=$v; }
                
                $gens = $conn->query("SELECT troops FROM generals WHERE user_id=$targetId AND x={$p['capital_x']} AND y={$p['capital_y']}");
                while($gen = $gens->fetch_assoc()) {
                    $gt = json_decode($gen['troops'], true) ?? [];
                    foreach ($gt as $k => $v) { if(!isset($totalTroops[$k])) $totalTroops[$k]=0; $totalTroops[$k]+=$v; }
                }
                
                $report .= "نیروهای پایتخت:\n" . formatLosses($totalTroops);
            }
            sendMessage($spyId, $report);
            $conn->query("UPDATE players SET spy_status='ready' WHERE user_id=$spyId");
        } else {
            $cooldown = date('Y-m-d H:i:s', strtotime("+" . SPY_FAILURE_RECOVERY_DAYS . " days"));
            $conn->query("UPDATE players SET spy_status='recovering', spy_cooldown='$cooldown' WHERE user_id=$spyId");
            sendMessage($spyId, "❌ **مأموریت شکست خورد!**\nجاسوس شما در قلمرو $targetName دستگیر شد.");
            sendMessage($targetId, "🚨 **نفوذ امنیتی!**\nیک جاسوس دشمن در قلمرو شما اعدام شد.");
        }
    }
    $conn->query("TRUNCATE TABLE espionage_missions");
}

function processDailyReportAndUpkeep() {
    global $conn;
    $players = $conn->query("SELECT user_id FROM players");
    
    while($p = $players->fetch_assoc()) {
        $uid = $p['user_id'];
        $report = "📅 **گزارش روزانه**\n------------------\n";
        
        // 1. تولید
        $bRes = $conn->query("SELECT type, level FROM buildings WHERE user_id=$uid");
        $levels = [];
        while($b = $bRes->fetch_assoc()) $levels[$b['type']] = $b['level'];
        
        $farmLvl = $levels['farm'] ?? 0; $mineLvl = $levels['mine'] ?? 0; $lumberLvl = $levels['lumber'] ?? 0;
        $prodFood = ($farmLvl > 0 ? $farmLvl * BUILDING_DATA['farm']['prod_rate'] : 10) * 24;
        $prodStone = ($mineLvl > 0 ? $mineLvl * BUILDING_DATA['mine']['prod_rate'] : 5) * 24;
        $prodWood = ($lumberLvl > 0 ? $lumberLvl * BUILDING_DATA['lumber']['prod_rate'] : 10) * 24;

        // 2. مصرف
        $totalTroops = 0;
        $userGenerals = [];
        $userGarrisons = [];

        $genRes = $conn->query("SELECT id, name, troops FROM generals WHERE user_id=$uid");
        while($gen = $genRes->fetch_assoc()) {
            $troops = json_decode($gen['troops'], true) ?? [];
            foreach($troops as $count) $totalTroops += $count;
            $userGenerals[] = $gen;
        }
        $tileRes = $conn->query("SELECT x, y, garrison FROM map_tiles WHERE owner_id=$uid AND type='capital'");
        while($tile = $tileRes->fetch_assoc()) {
            $garrison = json_decode($tile['garrison'], true) ?? [];
            foreach($garrison as $count) $totalTroops += $count;
            $userGarrisons[] = $tile;
        }

        $consumption = $totalTroops * 1; 
        $resRow = $conn->query("SELECT * FROM player_resources WHERE user_id=$uid")->fetch_assoc();
        
        $finalFood = $resRow['food'] + $prodFood - $consumption;
        $finalWood = $resRow['wood'] + $prodWood;
        $finalStone = $resRow['stone'] + $prodStone;
        $finalGold = $resRow['gold'];
        
        $isStarving = false;
        $deaths = 0;

        if ($finalFood < 0) {
            $isStarving = true;
            $finalFood = 0;
            
            foreach ($userGenerals as $gen) {
                $troops = json_decode($gen['troops'], true) ?? [];
                $newTroops = [];
                foreach ($troops as $t => $c) {
                    $survivors = floor($c * 0.90);
                    $deaths += ($c - $survivors);
                    if ($survivors > 0) $newTroops[$t] = $survivors;
                }
                $json = json_encode($newTroops);
                $conn->query("UPDATE generals SET troops='$json' WHERE id={$gen['id']}");
            }
            foreach ($userGarrisons as $tile) {
                $troops = json_decode($tile['garrison'], true) ?? [];
                $newTroops = [];
                foreach ($troops as $t => $c) {
                    $survivors = floor($c * 0.90);
                    $deaths += ($c - $survivors);
                    if ($survivors > 0) $newTroops[$t] = $survivors;
                }
                $json = json_encode($newTroops);
                $conn->query("UPDATE map_tiles SET garrison='$json' WHERE x={$tile['x']} AND y={$tile['y']}");
            }
        }

        $conn->query("UPDATE player_resources SET food=$finalFood, wood=$finalWood, stone=$finalStone, gold=$finalGold, last_update=NOW() WHERE user_id=$uid");

        $report .= "🌾 غذا: +$prodFood | 🪵 چوب: +$prodWood | 🪨 سنگ: +$prodStone\n";
        if ($consumption > 0) $report .= "🍽 مصرف ارتش: -$consumption غذا\n";
        if ($isStarving) $report .= "💀 **قحطی!** $deaths سرباز تلف شدند.\n";
        
        $qRes = $conn->query("SELECT building FROM building_queue WHERE user_id=$uid");
        if ($qRes->num_rows > 0) {
            $report .= "\n🏗 در حال ساخت:\n";
            while($q = $qRes->fetch_assoc()) $report .= "▫️ " . (BUILDING_DATA[$q['building']]['name'] ?? $q['building']) . "\n";
        }

        $report .= "\n💰 **موجودی:**\n🥇 $finalGold | 🌾 $finalFood | 🪵 $finalWood | 🪨 $finalStone";
        sendMessage($uid, $report);
    }
}

function checkLastStandStatus() {
    global $conn;
    $players = $conn->query("SELECT user_id, last_stand_turns FROM players");
    while($p = $players->fetch_assoc()) {
        $uid = $p['user_id'];
        $capSql = "SELECT COUNT(*) as cnt FROM map_tiles WHERE type='capital' AND owner_id=$uid";
        $capCount = $conn->query($capSql)->fetch_assoc()['cnt'];
        if ($capCount > 0) {
            if ($p['last_stand_turns'] !== null) {
                $conn->query("UPDATE players SET last_stand_turns = NULL WHERE user_id=$uid");
                sendMessage($uid, "✅ **نجات!** پایتخت پس گرفته شد.");
            }
        } else {
            if ($p['last_stand_turns'] === null) {
                $conn->query("UPDATE players SET last_stand_turns = 2 WHERE user_id=$uid");
                sendMessage($uid, "🚨 **سقوط پایتخت!**\nشما ۲ روز فرصت دارید تا یک قلعه تسخیر کنید، وگرنه ارتش منحل می‌شود.");
            } elseif ($p['last_stand_turns'] > 0) {
                $newTurn = $p['last_stand_turns'] - 1;
                $conn->query("UPDATE players SET last_stand_turns = $newTurn WHERE user_id=$uid");
                if ($newTurn == 0) {
                    $conn->query("DELETE FROM generals WHERE user_id=$uid");
                    $conn->query("UPDATE player_resources SET gold=0, food=0, wood=0, stone=0 WHERE user_id=$uid");
                    sendMessage($uid, "💀 **پایان امپراتوری!**\nارتش شما منحل شد. بازی را از نو شروع کنید.");
                } else {
                    sendMessage($uid, "⚠️ **وضعیت قرمز!** $newTurn روز تا نابودی کامل باقیست.");
                }
            }
        }
    }
}

// ------------------------------------------------------------------
// توابع محاسباتی (بدون ارسال پیام)
// ------------------------------------------------------------------

function formatLosses($losses) {
    if (empty($losses)) return "ناچیز";
    $parts = [];
    foreach ($losses as $k => $v) {
        $name = (defined('UNIT_STATS') && isset(UNIT_STATS[$k]['name'])) ? UNIT_STATS[$k]['name'] : $k;
        $parts[] = "$v $name";
    }
    return implode("، ", $parts);
}

function calculateBattle($attackers, $defenders) {
    $stats = UNIT_STATS;
    $bonuses = BATTLE_BONUSES;
    $attPower = 0; $attHP = 0;
    foreach ($attackers as $type => $count) {
        $dmg = $stats[$type]['att'] * $count;
        $attHP += $stats[$type]['hp'] * $count;
        foreach ($defenders as $defType => $defCount) {
            if (isset($bonuses[$type][$defType])) $dmg *= $bonuses[$type][$defType];
        }
        $attPower += $dmg;
    }
    $defPower = 0; $defHP = 0;
    foreach ($defenders as $type => $count) {
        $dmg = $stats[$type]['att'] * $count;
        $defHP += $stats[$type]['hp'] * $count;
        $dmg *= 1.2; 
        $defPower += $dmg;
    }
    $defHP *= 1.2;
    
    $rawAttLoss = ($attHP > 0) ? ($defPower / $attHP) : 1;
    $rawDefLoss = ($defHP > 0) ? ($attPower / $defHP) : 1;
    
    if ($rawAttLoss < $rawDefLoss) {
        $winner = 'attacker';
    } else {
        $winner = 'defender';
    }
    
    // قانون شکست سنگین (Rout Rule: حداقل 70 درصد تلفات برای بازنده)
    if ($winner == 'attacker') {
        $attLossPct = min(1, $rawAttLoss);
        $defLossPct = min(1, max($rawDefLoss, 0.70)); 
    } else {
        $attLossPct = min(1, max($rawAttLoss, 0.70)); 
        $defLossPct = min(1, $rawDefLoss);
    }

    $defSurvivors = [];
    foreach ($defenders as $t => $c) {
        $survivors = floor($c * (1 - $defLossPct));
        if ($survivors > 0) $defSurvivors[$t] = $survivors;
    }
    
    return [
        'winner' => ($winner),
        'defender_survivors' => $defSurvivors,
        'att_loss_pct' => $attLossPct,
        'def_loss_pct' => $defLossPct
    ];
}

function retreatGeneral($gen) {
    global $conn;
    $shifts = [[0,1], [0,-1], [1,0], [-1,0]];
    foreach ($shifts as $s) {
        $nx = $gen['x'] + $s[0];
        $ny = $gen['y'] + $s[1];
        if ($nx >= 1 && $nx <= MAP_SIZE && $ny >= 1 && $ny <= MAP_SIZE) {
            updateGeneralPos($gen['id'], $nx, $ny);
            return;
        }
    }
}

function updateGeneralPos($id, $x, $y) { global $conn; $conn->query("UPDATE generals SET x=$x, y=$y WHERE id=$id"); }
function updateGeneralTroops($id, $troops) { global $conn; $json = json_encode($troops); $conn->query("UPDATE generals SET troops='$json' WHERE id=$id"); }
?>